% This procedure computes the non-linear transition from the old steady
% state to the new steady state (with fracking), absent of shocks

clear all
close all
path(pathdef)
addpath ./functions/

% Setting parallel pool for speed. To set without parallel pool comment out
% the next four lines and set useparallel in line 98 to false
if isempty(gcp('nocreate'))
    myCluster = parcluster('local');
    parpool('local', myCluster.NumWorkers);
    clear myCluster
end



%% Get Initial Conditions
load ../../'Estimation Results'/Benchmark_Model.mat

beta    = parameters(1); 
th_n_ss = parameters(2);     
th_o_ss = parameters(3);     
MS_o    = parameters(4);
lambda  = parameters(5);
eta     = parameters(6);        
R_O     = parameters(7);
gamma   = parameters(8);
L       = parameters(9);
alpha   = parameters(10);
delta   = parameters(11);
nu      = parameters(12);

epsilon = parameters(13);
% psi     = parameters(14);
rho_a1  = parameters(14);
rho_a2  = parameters(15);
u_var   = parameters(16);    
a_var   = parameters(17);     

est_par_names = {'beta', 'epsilon', 'alpha', 'eta', 'lambda', 'nu', 'delta', 'gamma'};
for i = 1:length(est_par_names)
    eval(['estim_param.' est_par_names{i} ' = ' est_par_names{i} ';']);
end

calib_targets.theta_n = th_n_ss;
calib_targets.theta_o = th_o_ss;
calib_targets.MS_o    = MS_o;
calib_targets.R_o     = R_O;
calib_targets.L       = L;

[struct_param, endo_0, phis] = calibrate_ss(estim_param, calib_targets);

endo_0.phis = phis;

% Add fracking variables to initial conditions
endo_0.F_f          = 0.005;
endo_0.X_f          = 0;
endo_0.I_f          = 0;
endo_0.theta_f      = 0;
endo_0.mu_1_f       = 0;
endo_0.mu_2_f       = 0;
endo_0.phis.mu_1_f  = 0;
endo_0.phis.mu_2_f  = 0;
endo_0.phis.I_f     = 0;
endo_0.phis.X_f     = 0;
endo_0.phis.F_f     = 0;
endo_0.phis.theta_f = 0;

%% Get Terminal Conditions

exo_param.beta    = beta;
exo_param.epsilon = epsilon;
exo_param.alpha   = alpha;
exo_param.eta     = eta;
exo_param.psi     = psi;
exo_param.lambda  = lambda;
exo_param.nu      = nu;
exo_param.delta   = delta;
exo_param.gamma   = gamma;
exo_param.G_n     = struct_param.G_n;
exo_param.G_o     = struct_param.G_o;
exo_param.s_o     = struct_param.s_o;
exo_param.varphi  = struct_param.varphi;
exo_param.xi      = struct_param.xi;
exo_param.eta_f   = 2.23;
exo_param.lambda_f= 1;
MS_f              = 0.2030704;

evalc('th_n_ss = fsolve( @(tt) calibrate_ss_resid_fracking(tt, exo_param, MS_f), 0.03);');
[~, struct_param, endo_T, phis] = calibrate_ss_resid_fracking(th_n_ss, exo_param, MS_f);

endo_T.phis = phis;

%% Find transition

T = 300; % Years of transition from old to new steady state


options = optimoptions('fsolve', 'display', 'iter', 'MaxFunctionEvaluations', 10^6,'MaxIterations', 10^6,...
                       'Algorithm', 'trust-region', 'SubproblemAlgorithm', 'cg', 'UseParallel', true);

endo_0_orig = endo_0;
                   
guess_bnd = @(x) [exp(x(1:(3*T)))./(1+exp(x(1:(3*T)))); exp(x((3*T)+1:end))];
guess_inv = @(x) [log(x(1:(3*T))./(1-(x(1:(3*T))))); log(x((3*T)+1:end))];

% Start from terminal steady state
vec_guess = ones(1, T);
guess = nan(T, 5);
guess(:, 1) = endo_0.theta_n + (endo_T.theta_n-endo_0.theta_n)*vec_guess;
guess(:, 2) = endo_0.theta_o + (endo_T.theta_o-endo_0.theta_o)*vec_guess;
guess(:, 3) = endo_0.theta_f + (endo_T.theta_f-endo_0.theta_f)*vec_guess;
guess(:, 4) = endo_0.p       + (endo_T.p      -endo_0.p      )*vec_guess;
guess(:, 5) = endo_0.phis.RC + (endo_T.phis.RC-endo_0.phis.RC)*vec_guess;

guess(end, 2) = endo_T.phis.O;
x = guess_inv(guess(:));

zeta_0_ult = 0.0;
rho_zeta_ult = 0.88;

% Since the non-linear system is so large, we have to solve it slowly.
% Rather than starting immediatetly with the initial conditions being the
% old steady state, we start by solving the transition dynamics when the
% initial conditions are close to the terminal steady state. In particular,
% we set the initial condition to be old_ss * adj + new_ss * (1-adj). We
% start from a low value of adj and solve the transition. We then slowly
% increase the adjustment parameter and use the previous transition
% solution as the initial guess. When we get to adj=1, we have solved our
% full non-linear transition dynamics.

for adj = 0.05:0.05:1
    
    disp(['Solving iteration for adj = ' num2str(adj) '...']);

    endo_0.theta_n  = adj * endo_0_orig.theta_n  + (1-adj) * endo_T.theta_n;
    endo_0.X_n      = adj * endo_0_orig.X_n      + (1-adj) * endo_T.X_n;
    endo_0.F_n      = adj * endo_0_orig.F_n      + (1-adj) * endo_T.F_n;
    endo_0.theta_o  = adj * endo_0_orig.theta_o  + (1-adj) * endo_T.theta_o;
    endo_0.X_o      = adj * endo_0_orig.X_o      + (1-adj) * endo_T.X_o;
    endo_0.F_o      = adj * endo_0_orig.F_o      + (1-adj) * endo_T.F_o;
    endo_0.X_f      = adj * endo_0_orig.X_f      + (1-adj) * endo_T.X_f;
    endo_0.F_f      = adj * endo_0_orig.F_f      + (1-adj) * endo_T.F_f;
    endo_0.K        = adj * endo_0_orig.K        + (1-adj) * endo_T.K;

    endo_0.phis.theta_f = adj*endo_0_orig.phis.theta_f   + (1-adj)*endo_T.phis.theta_f;
    endo_0.phis.theta   = adj*endo_0_orig.phis.theta     + (1-adj)*endo_T.phis.theta;
    endo_0.phis.F       = adj*endo_0_orig.phis.F         + (1-adj)*endo_T.phis.F;
    endo_0.phis.X       = adj*endo_0_orig.phis.X         + (1-adj)*endo_T.phis.X;
    endo_0.phis.X_f     = adj*endo_0_orig.phis.X_f       + (1-adj)*endo_T.phis.X_f;
    endo_0.phis.F_f     = adj*endo_0_orig.phis.F_f       + (1-adj)*endo_T.phis.F_f;
    endo_0.phis.r       = adj*endo_0_orig.phis.r         + (1-adj)*endo_T.phis.r;
    
    zeta_0 = adj * zeta_0_ult + (1-adj);
    
    x = fsolve(@(x) full_transition_inv_tech_progress(guess_bnd(x), endo_0, endo_T, struct_param, zeta_0, rho_zeta_ult), x , options);
    [resid, TD_] = full_transition_inv_tech_progress(guess_bnd(x), endo_0, endo_T, struct_param, zeta_0, rho_zeta_ult);

end

delete(gcp('nocreate'))
save ./mat_files/TD_to_fracking

